/**
 * \file: mlink_wfd_decoder_plugin.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MLINK WiFi-Display Decoder plug-in
 *
 * \component: mlink
 *
 * \author: Ajay Kumar Sahoo ajaykumar.sahoo@in.bosch.com
 *
 * \copyright (c) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \history
 * 0.1 Ajay Kumar Sahoo Initial version
 *
 ***********************************************************************/

#ifndef DISABLE_WFD_PLUGIN
#include <mlink_wfd_decoder_plugin.h>
#include <gst/gst.h>
#include <gst/controller/gstcontroller.h>
#include <gst/base/gstbasesink.h>
#include <gst/app/gstappsrc.h>
#include <mlink_wfd_decoder_plugin_gstbuffer.h>
#include <uspi/EndpointDataSharing.h>

/* PRQA: Lint Message 826: deactivation because casting mechanism of GObject throws the finding */
/*lint -e826*/

#define DECODER_NAME "ADIT WiFi-Display GStreamer Decoder"
#define DECODER_VERSION "1.0"
#define DECODER_RECYCLE_RTP_PACKET 0xffff

using adit::uspi::SharedDataSender;

void WiFiDisplayDecoderRequestIDR(VNCWiFiDisplayDecoder *pDecoder)
{
	pDecoder->factoryInitInfo->pDecoderSupportingAPI->vncWiFiDisplayDecoderRequestIDR(pDecoder->decoderContext);
}

VNCWiFiDisplayDecoder* VNCCALL
WiFiDisplayDecoderCreate(VNCWiFiDisplayDecoderFactory *pDecoderFactory,
                         VNCWiFiDisplayDecoderContext  decoderContext)
{
    MLINK_UNUSED(pDecoderFactory);
    MLINK_UNUSED(decoderContext);
    /*Create VNCWiFiDisplayDecoderImpl object*/
    struct VNCWiFiDisplayDecoderImpl *decoder;

    decoder = (struct VNCWiFiDisplayDecoderImpl *)malloc(sizeof(struct VNCWiFiDisplayDecoderImpl));
    if(decoder == NULL)
    {
	pDecoderFactory->pDecoderSupportingAPI->vncWiFiDisplayDecoderLog(decoderContext,0,"Failed to allocate memory.Decoder cannot be created");
        return NULL;
    }
    memset(decoder, 0 , sizeof(struct VNCWiFiDisplayDecoderImpl));

    /*Initialize decoder*/
    decoder->decoderContext=decoderContext;
    decoder->factoryInitInfo=pDecoderFactory;


    return (VNCWiFiDisplayDecoder*)decoder;
}

void VNCCALL
WiFiDisplayDecoderDestroy(VNCWiFiDisplayDecoder *pDecoder )
{
    MLINK_UNUSED(pDecoder);
    if(pDecoder)
    {
        //unref the pipeline here.
        if (pDecoder->factoryInitInfo->pipeline)
        {
            g_clear_pointer (&pDecoder->factoryInitInfo->pipeline, gst_object_unref);
            WiFiDisplayDecoderLogInfo(pDecoder,"Gstreamer unreferenced");
        }
        free(pDecoder);
        pDecoder = NULL;
    }

}

VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderFactoryGetSupportInfo(VNCWiFiDisplayDecoderFactory *pDecoderFactory,
                                        VNCWiFiDisplayDecoderSupportInfo *pDecoderSupportInfo,
                                        size_t decoderSupportInfoSize)
{
    MLINK_UNUSED(pDecoderFactory);
    MLINK_UNUSED(pDecoderSupportInfo);
    MLINK_UNUSED(decoderSupportInfoSize);

    const VNCWiFiDisplayDecoderSupportingAPI *sapi = pDecoderFactory->pDecoderSupportingAPI;
    memset(pDecoderSupportInfo, 0, decoderSupportInfoSize);

    if(decoderSupportInfoSize < sizeof(VNCWiFiDisplayDecoderSupportInfo))
    {
        //factory->logError("VNCWiFiDisplayDecoderSupportInfo too small -- SDK too old!");
        //todo:use sdk logging which will be directed to DLT
        pDecoderFactory->pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(pDecoderFactory->decoderFactoryContext,10,"VNCWiFiDisplayDecoderSupportInfo too small");
        return VNCWiFiDisplayDecoderErrorInvalidParameter;
    }
    pDecoderSupportInfo->pDecoderName = sapi->vncWiFiDisplayDecoderFactoryAllocString(pDecoderFactory->decoderFactoryContext,
                                                                                      (const char* )DECODER_NAME);
    pDecoderSupportInfo->pDecoderVersion = sapi->vncWiFiDisplayDecoderFactoryAllocString(pDecoderFactory->decoderFactoryContext,
                                                                                         (const char* )DECODER_VERSION);
    pDecoderSupportInfo->disableInternalRTPBuffering = FALSE;

    if(!pDecoderSupportInfo->pDecoderName || !pDecoderSupportInfo->pDecoderVersion)
    {
        sapi->vncWiFiDisplayDecoderFactoryFree(pDecoderFactory->decoderFactoryContext, pDecoderSupportInfo->pDecoderName);
        sapi->vncWiFiDisplayDecoderFactoryFree(pDecoderFactory->decoderFactoryContext, pDecoderSupportInfo->pDecoderVersion);

        return VNCWiFiDisplayDecoderErrorNoMemory;
    }

    return VNCWiFiDisplayDecoderErrorNone;

}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderGetEventHandles(VNCWiFiDisplayDecoder *pDecoder,
                                  const VNCWiFiDisplayDecoderEventHandle **ppEventHandles,
                                  const VNCWiFiDisplayDecoderEventDirection **ppEventDirections,
                                  size_t *eventHandleCount)
{
    MLINK_UNUSED(pDecoder);
    MLINK_UNUSED(ppEventHandles);
    MLINK_UNUSED(ppEventDirections);
    MLINK_UNUSED(eventHandleCount);

    *eventHandleCount = 0; //No event handles from this decoder
    return VNCWiFiDisplayDecoderErrorNone;
}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderActivity(VNCWiFiDisplayDecoder *pDecoder)
{
    MLINK_UNUSED(pDecoder);
    return VNCWiFiDisplayDecoderErrorNone;
}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderPropertySet(VNCWiFiDisplayDecoder *pDecoder,
                              VNCWiFiDisplayDecoderProperty propertyKey,
                              const void *newValue,
                              size_t valueSize)
{
    MLINK_UNUSED(pDecoder);
    MLINK_UNUSED(propertyKey);
    MLINK_UNUSED(newValue);
    MLINK_UNUSED(valueSize);

    return VNCWiFiDisplayDecoderErrorNone;
}

static void WiFiDisplayDecodeOnPadAddedToSink(GstElement *element,GstPad *pad,gpointer data)
{
    MLINK_UNUSED(element);
    VNCWiFiDisplayDecoder *pDecoder=(VNCWiFiDisplayDecoder *)data;
    GstElement *pipeline = pDecoder->factoryInitInfo->pipeline;
    GstPad *sinkpad = NULL;
    GstPadLinkReturn ret;
    GstElement *audio = gst_bin_get_by_name(GST_BIN (pipeline),"wfd_audiosink");
    WiFiDisplayDecoderLogInfo(pDecoder,"Decodebin2 pad-added callback triggered");
    if(audio != NULL)
    {
        sinkpad = gst_element_get_static_pad (audio, "sink");
        if(sinkpad != NULL )
        {
            if (!GST_PAD_IS_LINKED (sinkpad))
            {
                ret = gst_pad_link (pad, sinkpad);
                if(ret != GST_PAD_LINK_OK)
                {
                    WiFiDisplayDecoderLogError(pDecoder,"Linking failed.Could not link decodebin2 to alsasink");
                }
            }
            else
            {
                WiFiDisplayDecoderLogWarn(pDecoder,"alsasink sinkpad already linked");
            }
        }
        else
        {
            WiFiDisplayDecoderLogError(pDecoder,"Failed to get alsasink static sinkpad");
        }
    }
    else
    {
        WiFiDisplayDecoderLogError(pDecoder,"alsasink element not found");
    }
    g_clear_pointer (&audio, g_object_unref);
    g_clear_pointer (&sinkpad, g_object_unref);
}

static void
WiFiDisplayDecodeOnPadAdded (GstElement *element,
                             GstPad     *pad,
                             gpointer    data)
{
    MLINK_UNUSED(element);
    GstCaps *caps;
    GstStructure *str;
    VNCWiFiDisplayDecoder *pDeoder=(VNCWiFiDisplayDecoder *)data;
    GstElement *pipeline = pDeoder->factoryInitInfo->pipeline;
    GstPad *sinkpad = NULL;
    GstPadLinkReturn ret;
    /* check media type */
    caps = gst_pad_get_caps (pad);
    str = gst_caps_get_structure (caps, 0);
    const char *capname = gst_structure_get_name (str);
    WiFiDisplayDecoderLogInfo(pDeoder,"aiurdemux pad-added callback triggered");
    WiFiDisplayDecoderLogInfo(pDeoder,capname);
    if (g_str_has_prefix (capname, "audio/mpeg"))
    {
        WiFiDisplayDecoderLogInfo(pDeoder,"WiFiDisplayDecodeOnPadAdded: AUDIO PAD");
        GstElement *aqueue = gst_bin_get_by_name(GST_BIN (pipeline), "wfd_audioqueue");
        sinkpad = gst_element_get_static_pad (aqueue, "sink");
    }
    else if(g_str_has_prefix (capname, "video/x-h264"))
    {
        WiFiDisplayDecoderLogInfo(pDeoder,"WiFiDisplayDecodeOnPadAdded: VIDEO PAD");
        GstElement *vqueue = gst_bin_get_by_name(GST_BIN (pipeline), "wfd_vedioqueue");
        sinkpad = gst_element_get_static_pad (vqueue, "sink");
    }
    else
    {
        WiFiDisplayDecoderLogError(pDeoder,"Unsupported capability pad");
        WiFiDisplayDecoderLogError(pDeoder,capname);
    }

    if(sinkpad != NULL )
    {
        if (!GST_PAD_IS_LINKED (sinkpad))
        {
            ret = gst_pad_link (pad, sinkpad);
            if(ret != GST_PAD_LINK_OK)
            {
                WiFiDisplayDecoderLogError(pDeoder,"Link to queue sink failed");
            }
        }
        else
        {
            WiFiDisplayDecoderLogWarn(pDeoder,"Queue sinkpad already linked");
        }
    }
    else
    {
        WiFiDisplayDecoderLogError(pDeoder,"Failed to get queue static sinkpad");
    }
    g_clear_pointer (&sinkpad, g_object_unref);
    g_clear_pointer (&caps, gst_caps_unref);
}

void  WiFiDisplayDecoderWfdBufferFinalize(GstWfdBuffer *wfdBuf)
{


    VNCWiFiDisplayDecoder *pDecoder  = wfdBuf->pDecoder;
    VNCWiFiDisplayRTPPacket *pPacket = wfdBuf->pPacket;

    const VNCWiFiDisplayDecoderSupportingAPI *sapi
                      = pDecoder->factoryInitInfo->pDecoderSupportingAPI;
    vnc_uint32_t messageType = DECODER_RECYCLE_RTP_PACKET;
    void *pData = (void *)pPacket;

    sapi->vncWiFiDisplayDecoderPostRTPThreadMessage(pDecoder->decoderContext, messageType, pData, sizeof(VNCWiFiDisplayRTPPacket));


}

GstWfdBuffer * WiFiDisplayDecoderWfdBufferNew( VNCWiFiDisplayDecoder *pDecoder,
                                               VNCWiFiDisplayRTPPacket *pPacket )
{
    GstWfdBuffer *wfdBuf;

    wfdBuf = GST_WFD_BUFFER(gst_mini_object_new (GST_TYPE_WFD_BUFFER));

    GST_BUFFER_DATA(wfdBuf) = pPacket->payloadData;
    GST_BUFFER_SIZE(wfdBuf) = pPacket->payloadLength;
    wfdBuf->pDecoder = pDecoder;
    wfdBuf->pPacket  = pPacket;
    wfdBuf->finalize = &WiFiDisplayDecoderWfdBufferFinalize;

    GST_LOG("Created new GstWfdBuffer: %p", wfdBuf);

    WiFiDisplayDecoderLogVerbose(pDecoder,"Created new GstWfdBuffer");
    return wfdBuf;
}

static void
WiFiDisplayDecoderPrepareBuffer(VNCWiFiDisplayDecoder *pDecoder,
                                GstAppSrc* appsrc,
                                VNCWiFiDisplayRTPPacket* pPacket)
{
    GstWfdBuffer * wfdBuffer;
    GstFlowReturn ret;

    GMainLoop  *mainloop = pDecoder->factoryInitInfo->mainloop;

    wfdBuffer = WiFiDisplayDecoderWfdBufferNew(pDecoder, pPacket);
    ret =  gst_app_src_push_buffer(appsrc, GST_BUFFER(wfdBuffer));
    if (ret != GST_FLOW_OK)
    {
        /* something wrong, stop pushing */
        WiFiDisplayDecoderLogWarn(pDecoder,"Stop pushing");
        g_main_loop_quit (mainloop);
    }
}


//static void
//WiFiDisplayDecodeOnAppSrcNeedData (GstElement *appsrc,
//        guint       unused_size,
//        gpointer    user_data)
//{  prepare_buffer((GstAppSrc*)appsrc);
//   g_print("***** cw ****** cb_need_data called ");
//   want = 1;
//   printf("# need_data 1\n");  //possible threading issue
//}

static GstBusSyncReply
WiFiDisplayDecodeOnBusMesg (GstBus * bus, GstMessage * message, gpointer user_datap)
{
    MLINK_UNUSED(bus);
    MLINK_UNUSED(user_datap);
    VNCWiFiDisplayDecoderImpl *pDecoder = (VNCWiFiDisplayDecoderImpl*)user_datap;
    WiFiDisplayDecoderLogVerbose(pDecoder,"Bus_message: called");
    WiFiDisplayDecoderLogInfo(pDecoder,gst_message_type_get_name (GST_MESSAGE_TYPE (message)));
//    switch (GST_MESSAGE_TYPE (message)) {
//    case GST_MESSAGE_ERROR: {
//        GError *err = NULL;
//        gchar *dbg_info = NULL;
//
//        gst_message_parse_error (message, &err, &dbg_info);
//        g_printerr ("ERROR from element %s: %s\n",
//                GST_OBJECT_NAME (message->src), err->message);
//        g_printerr ("Debugging info: %s\n", (dbg_info) ? dbg_info : "none");
//        g_error_free (err);
//        g_free (dbg_info);
//        g_main_loop_quit (loop);
//        break;
//    }
//    case GST_MESSAGE_EOS:
//        g_main_loop_quit (loop);
//        break;
//    default:
//        break;
//    }
    return GST_BUS_PASS;
}

void WiFiDisplayDecoderFirstVideoFrameRendered (GstElement* object,
                              GstPad* arg0,
                              gpointer user_data)

{
    MLINK_UNUSED(object);
    MLINK_UNUSED(arg0);
    MLINK_UNUSED(user_data);
    VNCWiFiDisplayDecoder *pDecoder = (VNCWiFiDisplayDecoder*)user_data;
    pDecoder->factoryInitInfo->decoderViewerContext->firstvideoframerendered(pDecoder->factoryInitInfo->decoderViewerContext->context);
    WiFiDisplayDecoderLogInfo(pDecoder,"WiFi Display Decoder First Video Frame Rendered");
    SharedDataSender::instance().transmitSurfaceData(pDecoder->factoryInitInfo->decoderViewerContext->context,
                                                 pDecoder->factoryInitInfo->display);
}

VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderStreamStart(VNCWiFiDisplayDecoder *pDecoder)
{
    GstBus *gstbus;
    //gchar  *audiostr = "alsasink device=Entertainment" ;
    // setup pipeline
    GstElement *pipeline = gst_pipeline_new ("pipeline");
    GstElement *appsrc   = gst_element_factory_make ("appsrc", "wfd_appsource");
    GstElement *typefind   = gst_element_factory_make ("typefind", "wfd_typefind");
    GstElement *demux   = gst_element_factory_make ("aiurdemux", "wfd_demux");
    GstElement *audioqueue   = gst_element_factory_make ("queue", "wfd_audioqueue");
    GstElement *decodebin2  = gst_element_factory_make ("decodebin2", "wfd_decodebin2");
    GstElement *audiosink  = gst_element_factory_make ("alsasink", "wfd_audiosink");
    GstElement *videoqueue   = gst_element_factory_make ("queue", "wfd_vedioqueue");
    GstElement *vpudec  = gst_element_factory_make ("vpudec", "wfd_vpudec");
    GstElement *videosink  = gst_element_factory_make ("gst_apx_sink", "wfd_videosink");
    //GstElement *audioqueue = gst_element_factory_make ("queue2", "wfd_audioqueue");
    //GstElement *videoqueue = gst_element_factory_make ("queue2", "wfd_videoqueue");
    WiFiDisplayDecoderLogInfo(pDecoder,"WiFiDisplayDecoder Stream Started");

    //GstElement *audiosink  = gst_element_factory_make ("filesink", "wfd_audiosink");
//    GError * error = NULL;
//    GstElement * audiosink = gst_parse_launch(audiostr, &error);
//    if (error)
//    {
//        printf("Error parsing audio string: %s",(error->message));
//        g_error_free(error);
//    }

    if (!pipeline || !appsrc || !typefind || !demux || !audioqueue || !decodebin2 || !audiosink
         || !vpudec || !videoqueue || !videosink)
    {
      //DLT Logging
        WiFiDisplayDecoderLogError(pDecoder,"Not all elements could be created");
        return VNCWiFiDisplayDecoderErrorInternal;
    }

    /*Register Bus callback*/
    gstbus = gst_pipeline_get_bus(GST_PIPELINE(pipeline));
    gst_bus_set_sync_handler(gstbus, WiFiDisplayDecodeOnBusMesg, pDecoder);
    g_clear_pointer (&gstbus, gst_object_unref);


    pDecoder->factoryInitInfo->pipeline = pipeline;
    pDecoder->factoryInitInfo->appsrc   = appsrc;

    g_object_set (G_OBJECT (appsrc), "is-live", TRUE, NULL);
    g_object_set (G_OBJECT (demux), "streaming-latency", (guint)0 , NULL);

    /*g_object_set (G_OBJECT (appsrc), "is-live", TRUE, NULL);
    g_object_set (G_OBJECT (demux), "streaming-latency", (guint)0 , NULL);*/
    //g_signal_connect (appsrc, "need-data", G_CALLBACK (WiFiDisplayDecodeOnAppSrcNeedData), NULL);

    //configure vpudec
    g_object_set (G_OBJECT (vpudec), "low-latency",TRUE, NULL);
    g_object_set (G_OBJECT (vpudec), "framedrop",FALSE, NULL);
    g_object_set (G_OBJECT (vpudec), "frame-plus",(gint)2, NULL);
    g_object_set (G_OBJECT (vpudec), "framerate-nu",(gint)60, NULL);
    g_object_set (G_OBJECT (vpudec), "dis-reorder",TRUE, NULL);

    //configure gst_apx_sink
    g_object_set (G_OBJECT (videosink), "sync", FALSE , NULL);
    g_object_set (G_OBJECT (videosink), "qos", FALSE , NULL);
    g_object_set (G_OBJECT (videosink), "max-lateness",(gint64)-1, NULL);
    g_object_set (G_OBJECT (videosink), "display-width",(guint32)1024, NULL);
    g_object_set (G_OBJECT (videosink), "display-height",(guint32)768, NULL);
    g_object_set (G_OBJECT (videosink), "layer-id",(guint32)3000, NULL);
    g_object_set (G_OBJECT (videosink), "surface-id",(guint32)40, NULL);
    g_object_set (G_OBJECT (videosink), "force-aspect-ratio",TRUE, NULL);
    void * display = NULL;
    g_object_get(G_OBJECT (videosink), "wl_display", &display, NULL);
    pDecoder->factoryInitInfo->display = (wl_display*)display;

    g_signal_connect(videosink,
                     "first-videoframe-rendered",
                     G_CALLBACK(WiFiDisplayDecoderFirstVideoFrameRendered),
                     pDecoder);

    //configure alsa sink
    g_object_set (G_OBJECT (audiosink), "device", "entertainment_main" , NULL);
    g_object_set (G_OBJECT (audiosink), "sync", FALSE , NULL);
//   g_object_set (G_OBJECT (audiosink), "blocksize", (guint32)400 , NULL);
//    g_object_set (G_OBJECT (audiosink), "location", "/tmp/wfdaudio.raw" , NULL);



    g_signal_connect(demux, "pad-added", G_CALLBACK(WiFiDisplayDecodeOnPadAdded), pDecoder);
    /* decodebin2 src pad is a sometimes pad - it gets created dynamically */
    g_signal_connect(decodebin2,"pad-added", G_CALLBACK(WiFiDisplayDecodeOnPadAddedToSink), pDecoder);

    gst_bin_add_many (GST_BIN (pipeline), appsrc, typefind, demux, audioqueue,decodebin2, audiosink, videoqueue, vpudec, videosink, NULL);
    gst_element_link_many (appsrc, typefind, demux, NULL);
    gst_element_link_many (audioqueue,decodebin2,NULL);
    gst_element_link_many (videoqueue, vpudec, videosink, NULL);

    //gst_element_link (audioqueue, audiosink);
    WiFiDisplayDecoderLogInfo(pDecoder,"Gstreamer pipeline parsed successful");
    /* run */
    gst_element_set_state (pipeline, GST_STATE_PLAYING);
    //  g_main_loop_run (loop);
    WiFiDisplayDecoderLogInfo(pDecoder,"Gstreamer pipeline running");

    WiFiDisplayDecoderRequestIDR (pDecoder);

    return VNCWiFiDisplayDecoderErrorNone;
}

VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderStreamStop(VNCWiFiDisplayDecoder *pDecoder)
{
    gst_element_set_state (pDecoder->factoryInitInfo->pipeline, GST_STATE_NULL);
    SharedDataSender::instance().notifySurfaceDataExpiration((void*)0x12345678);
    WiFiDisplayDecoderLogVerbose(pDecoder,"WiFiDisplayDecoder Stream Stopped");
    return VNCWiFiDisplayDecoderErrorNone;
}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderStreamPause(VNCWiFiDisplayDecoder *pDecoder)
{
    gst_element_set_state (pDecoder->factoryInitInfo->pipeline, GST_STATE_PAUSED);
    WiFiDisplayDecoderLogVerbose(pDecoder,"WiFiDisplayDecoder Stream Paused");
    return VNCWiFiDisplayDecoderErrorNone;
}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderStreamResume(VNCWiFiDisplayDecoder *pDecoder)
{
	WiFiDisplayDecoderRequestIDR (pDecoder);
    gst_element_set_state (pDecoder->factoryInitInfo->pipeline, GST_STATE_PLAYING);
    WiFiDisplayDecoderLogVerbose(pDecoder,"WiFiDisplayDecoder Stream Resumed");
    return VNCWiFiDisplayDecoderErrorNone;
}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderStreamCleanup(VNCWiFiDisplayDecoder *pDecoder)
{
    WiFiDisplayDecoderLogVerbose(pDecoder,"WiFiDisplayDecoder Stream Cleanup");
    //tbd:what needs to be done here ?
    return VNCWiFiDisplayDecoderErrorNone;
}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderOnRTPPacket(VNCWiFiDisplayDecoder *pDecoder,
                              VNCWiFiDisplayRTPPacket *pPacket)
{
    WiFiDisplayDecoderLogVerbose(pDecoder,"WiFiDisplayDecoderOnRTPPacket: called");
    MLINK_UNUSED(pPacket);

    GstElement *appsrc = pDecoder->factoryInitInfo->appsrc;

    WiFiDisplayDecoderPrepareBuffer(pDecoder,(GstAppSrc*)appsrc,pPacket);

    g_main_context_iteration(g_main_context_default(),FALSE);
    return VNCWiFiDisplayDecoderErrorNone;
}
vnc_uint64_t VNCCALL
WiFiDisplayDecoderHandleRTPThreadMessage(VNCWiFiDisplayDecoder *pDecoder,
                                         vnc_uint32_t messageType,
                                         const void *pData,
                                         vnc_size_t dataSize)
{
    MLINK_UNUSED(dataSize);
    WiFiDisplayDecoderLogVerbose(pDecoder,"WiFiDisplayDecoder handling RTP ThreadMessage");

    if (messageType == DECODER_RECYCLE_RTP_PACKET)
    {
        VNCWiFiDisplayRTPPacket *pPacket = (VNCWiFiDisplayRTPPacket*)pData;
        pDecoder->factoryInitInfo->pDecoderSupportingAPI->vncWiFiDisplayDecoderRecycleRTPPacket(pDecoder->decoderContext, pPacket);
    }
    return 0;
}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderH264CodecSelectedCallback(VNCWiFiDisplayDecoder *pDecoder,
                                            const VNCWiFiDisplayH264Codec *pCodec,
                                            size_t codecSize)
{
    WiFiDisplayDecoderLogInfo(pDecoder,"WiFiDisplayDecoder H264Codec Selected");
    MLINK_UNUSED(pCodec);
    MLINK_UNUSED(codecSize);
    return VNCWiFiDisplayDecoderErrorNone;
}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderAudioCodecSelectedCallback(VNCWiFiDisplayDecoder *pDecoder,
                                             const VNCWiFiDisplayAudioCodec *pCodec,
                                             size_t codecSize)
{
    WiFiDisplayDecoderLogInfo(pDecoder,"WiFiDisplayDecoder AudioCodec Selected");
    MLINK_UNUSED(pCodec);
    MLINK_UNUSED(codecSize);
    return VNCWiFiDisplayDecoderErrorNone;

}
VNCWiFiDisplayDecoderError VNCCALL
WiFiDisplayDecoderAVFormatChangeTimingCallback(VNCWiFiDisplayDecoder *pDecoder,
                                               vnc_uint64_t PTS,
                                               vnc_uint64_t DTS)
{
    WiFiDisplayDecoderLogInfo(pDecoder,"WiFiDisplayDecoderAVFormatChangeTimingCallback  triggered");
    MLINK_UNUSED(PTS);
    MLINK_UNUSED(DTS);
    return VNCWiFiDisplayDecoderErrorNone;
}
void VNCCALL
WiFiDisplayDecoderFactoryTerminate(VNCWiFiDisplayDecoderFactory *pDecoderFactory)
{
    pDecoderFactory->pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(pDecoderFactory->decoderFactoryContext,30,"WiFiDisplayDecoderFactoryTerminate: called");
    MLINK_UNUSED(pDecoderFactory);
    if(pDecoderFactory)
    {
        free(pDecoderFactory);
        pDecoderFactory =NULL;
    }
}

void WiFiDisplayDecoderInitializeInterface(VNCWiFiDisplayDecoderInterface *pDecoderInterface)
{
    pDecoderInterface->vncWiFiDisplayDecoderCreate = WiFiDisplayDecoderCreate;
    pDecoderInterface->vncWiFiDisplayDecoderDestroy = WiFiDisplayDecoderDestroy;
    pDecoderInterface->vncWiFiDisplayDecoderFactoryGetSupportInfo = WiFiDisplayDecoderFactoryGetSupportInfo;
    pDecoderInterface->vncWiFiDisplayDecoderGetEventHandles = WiFiDisplayDecoderGetEventHandles;
    pDecoderInterface->vncWiFiDisplayDecoderActivity = WiFiDisplayDecoderActivity;
    pDecoderInterface->vncWiFiDisplayDecoderPropertySet = WiFiDisplayDecoderPropertySet;
    pDecoderInterface->vncWiFiDisplayDecoderStreamStart = WiFiDisplayDecoderStreamStart;
    pDecoderInterface->vncWiFiDisplayDecoderStreamStop = WiFiDisplayDecoderStreamStop;
    pDecoderInterface->vncWiFiDisplayDecoderStreamPause = WiFiDisplayDecoderStreamPause;
    pDecoderInterface->vncWiFiDisplayDecoderStreamResume = WiFiDisplayDecoderStreamResume;
    pDecoderInterface->vncWiFiDisplayDecoderStreamCleanup = WiFiDisplayDecoderStreamCleanup;
    pDecoderInterface->vncWiFiDisplayDecoderOnRTPPacket = WiFiDisplayDecoderOnRTPPacket;
    pDecoderInterface->vncWiFiDisplayDecoderHandleRTPThreadMessage = WiFiDisplayDecoderHandleRTPThreadMessage;
    pDecoderInterface->vncWiFiDisplayDecoderH264CodecSelectedCallback = WiFiDisplayDecoderH264CodecSelectedCallback;
    pDecoderInterface->vncWiFiDisplayDecoderAudioCodecSelectedCallback = WiFiDisplayDecoderAudioCodecSelectedCallback;
    pDecoderInterface->vncWiFiDisplayDecoderAVFormatChangeTimingCallback = WiFiDisplayDecoderAVFormatChangeTimingCallback;
    pDecoderInterface->vncWiFiDisplayDecoderFactoryTerminate = WiFiDisplayDecoderFactoryTerminate;

}

VNCWiFiDisplayDecoderError
WiFiDisplayDecoderAddSupportedAudioCodec(const VNCWiFiDisplayDecoderSupportingAPI *pDecoderSupportingAPI,
                                        VNCWiFiDisplayDecoderFactoryContext *decoderFactoryContext)
{
     VNCWiFiDisplayDecoderError decoderError = VNCWiFiDisplayDecoderErrorNone;
     VNCWiFiDisplayAudioCodec   audioCodec;
     memset(&audioCodec, 0, sizeof(audioCodec));

     audioCodec.format = VNCWiFiDisplayAudioFormatLPCM;
     audioCodec.modes = VNCWiFiDisplayAudioModeLPCM_48000_16_2;
     audioCodec.latency = 20;
     /*Tell SDK what Audio codecs this G-streamer decoder supports*/
     decoderError = pDecoderSupportingAPI->vncWiFiDisplayDecoderAddAudioCodec(*decoderFactoryContext,
                                                                              (const VNCWiFiDisplayAudioCodec*) &audioCodec,
                                                                               sizeof(VNCWiFiDisplayAudioCodec));
     if (decoderError != VNCWiFiDisplayDecoderErrorNone)
     {
         pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(*decoderFactoryContext,5,"Failed to add LPCM audio codec");
         return decoderError;
     }

     memset(&audioCodec, 0, sizeof(audioCodec));
     audioCodec.format = VNCWiFiDisplayAudioFormatAC3;
     audioCodec.modes = VNCWiFiDisplayAudioModeAC3_48000_16_2;
     audioCodec.latency = 20;

     decoderError = pDecoderSupportingAPI->vncWiFiDisplayDecoderAddAudioCodec(*decoderFactoryContext,
                                                                              (const VNCWiFiDisplayAudioCodec*) &audioCodec,
                                                                               sizeof(VNCWiFiDisplayAudioCodec));
     if (decoderError != VNCWiFiDisplayDecoderErrorNone)
     {
         pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(*decoderFactoryContext,5,"Failed to add AC3 audio codec");
         return decoderError;
     }

     memset(&audioCodec, 0, sizeof(audioCodec));
     audioCodec.format = VNCWiFiDisplayAudioFormatAAC;
     audioCodec.modes = VNCWiFiDisplayAudioModeAAC_48000_16_2;
     audioCodec.latency = 20;

     decoderError = pDecoderSupportingAPI->vncWiFiDisplayDecoderAddAudioCodec(*decoderFactoryContext,
                                                                              (const VNCWiFiDisplayAudioCodec*) &audioCodec,
                                                                               sizeof(VNCWiFiDisplayAudioCodec));
     if (decoderError != VNCWiFiDisplayDecoderErrorNone)
     {
         pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(*decoderFactoryContext,5,"Failed to add AAC audio codec");
         return decoderError;
     }

     return decoderError;
}

VNCWiFiDisplayDecoderError
WiFiDisplayDecoderAddSupportedH264Codec(const VNCWiFiDisplayDecoderSupportingAPI *pDecoderSupportingAPI,
                                        VNCWiFiDisplayDecoderFactoryContext *decoderFactoryContext)
{
    VNCWiFiDisplayDecoderError decoderError = VNCWiFiDisplayDecoderErrorNone;
    VNCWiFiDisplayH264Codec h264Codec;
    memset(&h264Codec, 0, sizeof(h264Codec));

    h264Codec.profile = VNCWiFiDisplayH264ProfileCBP;
    h264Codec.level   = VNCWiFiDisplayH264Level_3_1;
    h264Codec.ceaSupportBitmap = VNCWiFiDisplayResolutionCEA_640_480_p60;
    h264Codec.hhSupportBitmap  = VNCWiFiDisplayResolutionHH_800_480_p30;
    h264Codec.frameSkippingSupported   = TRUE;
    h264Codec.frameSkippingMaxInterval = 0;
    pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(*decoderFactoryContext,30,"Supported H64 Codec added");
    /*Tell SDK what h.264 codecs this G-streamer decoder supports */
    decoderError = pDecoderSupportingAPI->vncWiFiDisplayDecoderAddH264Codec(*decoderFactoryContext,
                                                                            (const VNCWiFiDisplayH264Codec*)&h264Codec,
                                                                            sizeof (VNCWiFiDisplayH264Codec));
    return decoderError;
}
extern "C" VNCWiFiDisplayDecoderFactory* VNCCALL
VNCWiFiDisplayDecoderFactoryInitialize(VNCWiFiDisplayDecoderFactoryContext decoderFactoryContext,
                                       VNCWiFiDisplayDecoderViewerContext decoderViewerContext,
                                       VNCWiFiDisplayDecoderInterface *pDecoderInterface,
                                       size_t decoderInterfaceSize,
                                       const VNCWiFiDisplayDecoderSupportingAPI *pDecoderSupportingAPI,
                                       size_t decoderSupportingAPISize)
{
    MLINK_UNUSED(decoderFactoryContext);
    MLINK_UNUSED(decoderViewerContext);
    MLINK_UNUSED(pDecoderInterface);
    MLINK_UNUSED(decoderInterfaceSize);
    MLINK_UNUSED(pDecoderSupportingAPI);
    MLINK_UNUSED(decoderSupportingAPISize);

    VNCWiFiDisplayDecoderError decoderError;


    /*
     * Create  DecoderFactory structure
     *
     * todo: free all resource when SDK calls VNCWiFiDisplayDecoderfactoryTerminate()
     */
    struct VNCWiFiDisplayDecoderFactoryImpl *wfddecoderfactory = NULL ;
    wfddecoderfactory = (struct VNCWiFiDisplayDecoderFactoryImpl *)malloc(sizeof(struct VNCWiFiDisplayDecoderFactoryImpl));
    if(wfddecoderfactory == NULL)
    {
        //todo:log error
        pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(decoderFactoryContext,0,"WFD Decoderfactory memory allocation failed");
        //decoder will not be initialized
        return NULL;
    }
    memset(wfddecoderfactory,0, sizeof(struct VNCWiFiDisplayDecoderFactoryImpl));
    /* Initialize DecoderFactory structure */
    wfddecoderfactory->decoderFactoryContext    = decoderFactoryContext;
    wfddecoderfactory->decoderViewerContext     = decoderViewerContext;
    wfddecoderfactory->pDecoderSupportingAPI    = pDecoderSupportingAPI;
    wfddecoderfactory->decoderSupportingAPISize = decoderSupportingAPISize;
    pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(decoderFactoryContext,30,"Initialise gstreamer");
    //gst_init(NULL, NULL);
    wfddecoderfactory->mainloop = g_main_loop_new (NULL, FALSE);
    pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(decoderFactoryContext,30,"Initialise gstreamer successful");
    /* Initialize Wi-Fi Display decoder Interface
     *
     * Interface APIs are defined in this decoder  plug-in
     * and called by Wi-Fi Display SDK
     */
    memset(pDecoderInterface, 0, decoderInterfaceSize);
    WiFiDisplayDecoderInitializeInterface(pDecoderInterface);

    /*
     * Add supported audio codec to Wi-Fi Display SDK
     */
    decoderError = WiFiDisplayDecoderAddSupportedAudioCodec (pDecoderSupportingAPI,
                                             &wfddecoderfactory->decoderFactoryContext);
    if (decoderError != VNCWiFiDisplayDecoderErrorNone)
    {
        pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(decoderFactoryContext,0,"Failed to add supported audio codec to SDK");
        free(wfddecoderfactory);
        wfddecoderfactory = NULL;
        return NULL;
    }
    pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(decoderFactoryContext,30,"Add supported audio codec to SDK: Success");

    /*
     * Add supported h.264 codec to Wi-Fi Display SDK
     */
    decoderError = WiFiDisplayDecoderAddSupportedH264Codec (pDecoderSupportingAPI,
                                            &wfddecoderfactory->decoderFactoryContext);
    if (decoderError != VNCWiFiDisplayDecoderErrorNone)
    {
	pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(decoderFactoryContext,0,"Failed to add supported h264 codec to SDK");
        free(wfddecoderfactory);
        wfddecoderfactory = NULL;
        return NULL;
    }
    pDecoderSupportingAPI->vncWiFiDisplayDecoderFactoryLog(decoderFactoryContext,30,"Add supported h264 codec to SDK: Success");
    return (VNCWiFiDisplayDecoderFactory* )wfddecoderfactory;
    //return NULL;
}

/*lint +e826*/
#endif
